/************************************************************************
 * NAME:	fs-trs80.h
 *
 * DESCR:	The structure and routines for the TRS80 generic file
 *		system;
 *
 * NOTES:	
 ************************************************************************/

/* isTRSDOS13 - this flag indicates that the floppy is a DD TRSDOS 1.3	*/
/*   (or like) image.  These images are indicated by odd sector numbers	*/
/*   that start with 1 as opposed to 0.					*/

struct fs_trs80 {
    struct floppy	*floppy;
    int			isTRSDOS13;
    int			dirtrack;	/* which track has the directory */
};

/* TRSDOS13 looks to be pretty crazy - not only does it start numbering	*/
/* with sector 1 (which is how we notice it) but it also uses only 5	*/
/* directory entries as opposed to 8 per sector.  It also uses 3 sectors*/
/* per Gran as opposed to 5 or 6.		Further, it appears	*/
/* NOT to have directory entries for BOOT sectors or the DIR itself...	*/
/* though the GAT looks to be right.  Here are some useful macros for	*/
/* dealing with it.							*/

#define DIRENT_COUNT(f)		((f->isTRSDOS13)?5:8)
#define DIRENT_SCALER(f)	((f->isTRSDOS13)?48:32)

/* DEFINES for decoding the first byte of the directory entry	*/
/*  Note that a pointer to the entry is given as the arg.	*/

/* NOTE about EOF - if the EOFBYTEPOS is zero, that means that	*/
/*   there are 256 bytes in the ending sector.  In that case,	*/
/*   the EOFSECTOR indicates the sector just BEYOND the one in	*/
/*   which the last byte is.  The thought (I think)	 	*/
/*   here is that the EOFBYTEPOS/EOFSECTOR work	together	*/
/*   to indicate the first position PAST the last byte of data.	*/
/*   Interestingly, if there is no data in the file, these two	*/
/*   bytes would be 0/0.  With 256 bytes, it is 0/1.		*/
/*   Note that these are the NEWDOS "reported" values.  On the	*/
/*   actual diskette, the EOFSECTOR is incremented by one.  In	*/
/*   the previous example, with 255 bytes, it will be 255/1	*/

#define DIRENT_PROTECTION_BITS		0x07
#define DIRENT_INVISIBLE_BITS		0x08
#define DIRENT_ASSIGNED_BITS		0x10
#define DIRENT_SYSTEM_BITS		0x40
#define DIRENT_EXTENDED_BITS		0x80

/* use these defines when setting options for a file	*/
/* they should be OR'd together to set all of the bits	*/

#define DIRENT_VISIBLE			0
#define DIRENT_INVISIBLE		DIRENT_INVISIBLE_BITS
#define DIRENT_ASSIGNED			DIRENT_ASSIGNED_BITS
#define DIRENT_NOTASSIGNED		0
#define DIRENT_SYSTEM			DIRENT_SYSTEM_BITS
#define DIRENT_NONSYSTEM		0
#define DIRENT_EXTENDED			DIRENT_EXTENDED_BITS
#define DIRENT_NORMAL			0

/* protection bits really isn't what it may seem to be	*/
/* these things are in increasing order such that a	*/
/* "higher" level implies all of the lower levels	*/

#define DIRENT_PROTECT_NOACCESS		0x07
#define DIRENT_PROTECT_EXEC		0x06
#define DIRENT_PROTECT_READ		0x05
#define DIRENT_PROTECT_WRITE		0x04
#define DIRENT_PROTECT_RENAME		0x02
#define DIRENT_PROTECT_KILL		0x01
#define DIRENT_PROTECT_CHANGE		0x00

#define DIRENT_OPTIONS(entry)		(entry[0])
#define DIRENT_SET_OPTIONS(entry,ops)	(entry[0] = ops)

#define DIRENT_PROTECTION(entry)	((unsigned char)(entry[0])&DIRENT_PROTECTION_BITS)
#define DIRENT_IS_INVISIBLE(entry)	((unsigned char)(entry[0])&DIRENT_INVISIBLE_BITS)
#define DIRENT_IS_ASSIGNED(entry)	((unsigned char)(entry[0])&DIRENT_ASSIGNED_BITS)
#define DIRENT_IS_SYSTEM(entry)		((unsigned char)(entry[0])&DIRENT_EXTENDED_BITS)
#define DIRENT_IS_EXTENDED(entry)	((unsigned char)(entry[0])&0x80)
#define DIRENT_EOFBYTEPOS(entry)	((unsigned char)(entry[3]))
#define   DIRENT_SET_EOFBYTEPOS(e,v)	(DIRENT_EOFBYTEPOS(e) = v)
#define DIRENT_LOGICALRECLEN(entry)	((unsigned char)(entry[4]))
#define DIRENT_NAME(entry)		(entry+5)
#define DIRENT_NAME_CMP(entry,name)	(strncmp(DIRENT_NAME(entry),name,11))
#define DIRENT_NAME_CP_OUT(entry,name)	(strncpy(name,DIRENT_NAME(entry),11))
#define DIRENT_NAME_CP_IN(entry,name)	(strncpy(DIRENT_NAME(entry),name,11))
#define DIRENT_EXTENSION(entry)		(entry+13)
#define DIRENT_PWHASH_UPDATE(entry)	((int)entry[16]+ (((int)entry[17])<<8))
#define   DIRENT_SET_PWHASH_UPDATE(e,v)	(e[16]=v&(unsigned)0x00ff, e[17]=(v&(unsigned)0xff00)>>8)
#define DIRENT_PWHASH_ACCESS(entry)	((int)entry[18]+ (((int)entry[19])<<8))
#define   DIRENT_SET_PWHASH_ACCESS(e,v) (e[18]=v&(unsigned)0x00ff, e[19]=(v&(unsigned)0xff00)>>8)
#define DIRENT_EOFSECTOR(entry)		((unsigned int)entry[20]+ (((unsigned int)entry[21])<<8))
#define   DIRENT_SET_EOFSECTOR(e,v)	(e[20] = (v)&0xff, e[21] = ((v)>>8)&0xff)

/* a BLANK_PASSWORD is one that is eight spaces */
/* note that TRSDOS 1.3 apparently has a different	*/
/* hashing scheme, in that they say the default password	*/
/* is "PASSWORD" but the hash doesn't work out using the	*/
/* "normal" hashing that EVERYONE else does.  So, to work	*/
/* around the problem for now, I use the password "CONDUIT ",	*/
/* which hashes to the right value given the current hashing	*/
/* schedule.  Note that "PRICING " would work too.		*/


#define BLANK_PASSWORD(f)	(f->isTRSDOS13?"CONDUIT ":"        ")

#define DIRENT_EXTENT_COUNT(f)		   (f->isTRSDOS13?13:5)

#define DIRENT_EXTENT_TRACK(e,n)	   ((int)e[22+n*2])
#define   DIRENT_EXTENT_SET_TRACK(e,n,v)   (e[22+n*2] = v)
#define DIRENT_EXTENT_RELSECT(e,n)	   (((int)e[22+n*2+1]& 0xe0)>>5)
#define   DIRENT_EXTENT_SET_RELSECT(e,n,v) (e[22+n*2+1] = (e[22+n*2+1]&0x1f) | ((v&0x07)<<5))
#define DIRENT_EXTENT_LENGTH(f,e,n)	   (((int)e[22+n*2+1]& 0x1f)+((f->isTRSDOS13)?0:1))
#define   DIRENT_EXTENT_SET_LENGTH(f,e,n,v)  (e[22+n*2+1] = (e[22+n*2+1]&0xe0) + v - ((f->isTRSDOS13)?0:1))

/*#define DIRENT_EXTENT_LENGTH(f,e,n)	   (((int)e[22+n*2+1]& 0x1f)+1) */
/*#define   DIRENT_EXTENT_SET_LENGTH(f,e,n,v)  (e[22+n*2+1] = (e[22+n*2+1]&0xe0) + v - 1)*/

#define DIRENT_EXTENT_DEC_DIRENT(e,n)		((int)e[22+n*2+1]& 0x07)
#define   DIRENT_EXTENT_SET_DEC_DIRENT(e,n,v)	(e[22+n*2+1] = (e[22+n*2+1]&0xf8) | (v&0x07))
#define DIRENT_EXTENT_DEC_ENTRY(e,n)		DIRENT_EXTENT_RELSECT(e,n)
#define   DIRENT_EXTENT_SET_DEC_ENTRY(e,n,v)	DIRENT_EXTENT_SET_RELSECT(e,n,v)

#define DIRENT_EXTENT_NORMAL		0
#define DIRENT_EXTENT_EXTENDED		254
#define DIRENT_EXTENT_END		255

#define DIRENT_IS_EXTENT_NORMAL(entry,n)	(DIRENT_EXTENT_TRACK(entry,n)==DIRENT_EXTENT_NORMAL)
#define   DIRENT_EXTENT_SET_NORMAL(entry,n)	(DIRENT_EXTENT_TRACK(entry,n)=DIRENT_EXTENT_NORMAL)
#define DIRENT_IS_EXTENT_END(entry,n)		(DIRENT_EXTENT_TRACK(entry,n)==DIRENT_EXTENT_END)
#define   DIRENT_EXTENT_SET_END(entry,n)	(DIRENT_EXTENT_TRACK(entry,n)=DIRENT_EXTENT_END)
#define DIRENT_IS_EXTENT_EXTENDED(entry,n)	(DIRENT_EXTENT_TRACK(entry,n)==DIRENT_EXTENT_EXTENDED)
#define   DIRENT_EXTENT_SET_EXTENDED(entry,n)	(DIRENT_EXTENT_TRACK(entry,n)=DIRENT_EXTENT_EXTENDED)

/****************************************************************/
/* HERE'S the old way these were done, kept just in case	*/
/****************************************************************/
/* #define DIRENT_SECTORS(entry)	(DIRENT_EOFSECTOR(entry)+(DIRENT_EOFBYTEPOS(entry)==0?0:1))
/* #define DIRENT_BYTES(entry)		(DIRENT_EOFSECTOR(entry)*256+DIRENT_EOFBYTEPOS(entry))
/****************************************************************/
/* NEW WAY							*/
/****************************************************************/
#define DIRENT_SECTORS(entry)		(DIRENT_EOFSECTOR(entry))
#define DIRENT_BYTES(entry)		(DIRENT_EOFSECTOR(entry)*256+DIRENT_EOFBYTEPOS(entry)-\
                                                    ((DIRENT_EOFBYTEPOS(entry)==0||DIRENT_EOFSECTOR(entry)==0)?0:256))
#define DIRENT_SET_EOFFIELDS(entry,bytes) ((DIRENT_SET_EOFBYTEPOS(entry,bytes%256)),\
                                           (DIRENT_SET_EOFSECTOR(entry,bytes/256+((bytes%256)?1:0))))

/* protection bits table:
      111 - no access                             - ------
      110 - execute only                          - --x---
      101 - read/execute                          - r-x---
      100 - write/read/execute                    - rwx---
      011 - NOT USED                              - ??????
      010 - rename/write/read/execute             - rwxn--
      001 - kill/rename/write/read/execute        - rwxnd-
      000 - kill/rename/write/read/execute/change - rwxndc
*****/

#define DIRENT_PROT_EXEC(entry)		(DIRENT_PROTECTION(entry)<=6)
#define DIRENT_PROT_READ(entry)		(DIRENT_PROTECTION(entry)<=5)
#define DIRENT_PROT_WRITE(entry)	(DIRENT_PROTECTION(entry)<=4)
#define DIRENT_PROT_RENAME(entry)	(DIRENT_PROTECTION(entry)<=2)
#define DIRENT_PROT_KILL(entry)		(DIRENT_PROTECTION(entry)<=1)
#define DIRENT_PROT_OTHER(entry)	(DIRENT_PROTECTION(entry)==0)
#define DIRENT_PROT_STRING(entry)	(DIRENT_PROTECTION(entry)==7)?"------":\
                                        (DIRENT_PROTECTION(entry)==6)?"--x---":\
                                        (DIRENT_PROTECTION(entry)==5)?"r-x---":\
                                        (DIRENT_PROTECTION(entry)==4)?"rwx---":\
                                        (DIRENT_PROTECTION(entry)==3)?"??????":\
                                        (DIRENT_PROTECTION(entry)==2)?"rwxn--":\
                                        (DIRENT_PROTECTION(entry)==1)?"rwxnd-":"rwxndc"

/* DirentLoc() - returns a pointer to data for the given	*/
/*		 directory entry.    Note that the		*/
/*    track # is usually 17.	Note that the entry_sect	*/
/*    is always given as the logical directory sector (which is	*/
/*    2 less than the equiv logical/physical sector)		*/
/*    f = struct fs_trs80 *				        */
/*    s = side #						*/
/*    t = track #						*/
/*    es = directory entry sector				*/
/*    e  = directory entry # within that sector			*/

#define DirentLoc(f,s,es,e)	(LogicalSector(f->floppy,s,f->dirtrack,es+2).data + e*DIRENT_SCALER(f))

/*  HITLOC(floppy,side,track,entry_sector,entry#)		*/
/*    Return a pointer to the hit byte for the given directory	*/
/*    sector and entry within that sector.  Note that the	*/
/*    track # is usually 17, and the directory sector is locked	*/
/*    to 1 (which is the HIT sector).  Note that the entry_sect	*/
/*    is always given as the logical directory sector (which is	*/
/*    2 less than the equiv logical/physical sector)		*/
/*    f = struct fs_trs80 *				        */
/*    s = side #						*/
/*    t = track #						*/
/*    es = directory entry sector				*/
/*    e  = directory entry # within that sector			*/

#define HITSECTOR	1

/* #define HITLOC(f,s,t,es,e)	(LogicalSector(f,s,t,1).data+(e<<5)+es) */
#define HITLOC(f,s,es,e)	(LogicalSector(f->floppy,s,f->dirtrack,HITSECTOR).data+((f->isTRSDOS13)?(es*DIRENT_COUNT(f)+e):(e*DIRENT_SCALER(f)+es)))

/*  GAT routines - the following defines deal with the GAT	*/
/*    (Granule Allocation Table).  The track # is usually 17,	*/
/*    and the sector is locked to 0 (the GAT sector).  Note	*/
/*    that the entry sector (es) is always given as the logical	*/
/*    directory sector (which is 2 less than the equiv logical/	*/
/*    physical sector).						*/
/*    f = struct fs_trs80 *				        */
/*    s = side #						*/
/*    t = track #						*/
/*    g = granule # (NOTE - need to see what GAT looks like for	*/
/*			a DD diskette, although I'll assume that*/
/*      		it is a logical extension of SD)	*/
/*								*/
/*   GransPerTrack() - returns the number of granules per track	*/
/*			by looking at the density of the disk	*/
/*   GATGranLoc() - returns a pointer to the particular byte	*/
/*		    where the given gran allocation bit is.	*/
/*   GATGranAlloc() - returns non-zero if the given granule has	*/
/*		      been allocated.				*/
/*   GATGranSet() - allocates this granule as in use		*/
/*   GATGranFree() - "frees" this granule for later use
/*   GATLockLoc() - just like GATGranLoc() except with the gran	*/
/*		    lock-out table.				*/
/*   GATLockSet() - just like GATGranAlloc() except with the	*/
/*		    gran lock-out table.			*/

#define GATSECTOR	0

#define GransPerTrack(f)	((f->floppy->encoding == WD_MFM)?(f->isTRSDOS13?6:3):2)
#define SectsPerGran(f)		((f->floppy->encoding == WD_MFM)?(f->isTRSDOS13?3:6):5)

#define GATGranLoc(f,s,g)	(LogicalSector(f->floppy,s,f->dirtrack,GATSECTOR).data+((g)/GransPerTrack(f)))
#define GATGranAlloc(f,s,g)	(*GATGranLoc(f,s,g) & (0x01<<((g)%GransPerTrack(f))))
#define GATGranSet(f,s,g)	(*GATGranLoc(f,s,g) |= (0x01<<((g)%GransPerTrack(f))))
#define GATGranFree(f,s,g)	(*GATGranLoc(f,s,g) &= ~(0x01<<((g)%GransPerTrack(f))))
#define GATLockLoc(f,s,g)	(GATGranLoc(f,s,g)+0x60)
#define GATLockSet(f,s,g)	(*GATLockLoc(f,s,g) & (0x01<<((g)%GransPerTrack(f))))

/* GranTrack() - given a relative granule, return its track #	*/
/* GranRelSect() - given a rel gran, return its rel sect start	*/
/* GranNumber() - given a track and relative sector, return gran number	*/

#define GranTrack(f,g)		(g/GransPerTrack(f))
#define GranRelSect(f,g)	(g%GransPerTrack(f))
#define GranNumber(f,t,rs)	(GransPerTrack(f)*(t)+(rs))

/*  GATUnknown() - a pointer to 3 bytes of unknown function	*/
/*  GATMPWHash() - a pointer to 2 bytes representing the hash	*/
/*		   code for the master disk password		*/
/*  GATDiskName() - a pointer to the 16 bytes of name & date	*/
/*  GATAuto() - a pointer to 32 bytes of the auto program, if	*/
/*		0xe0 contains 0x0d then the command won't run	*/
/*  GATAutoRun() - returns 0 if the Auto program won't run	*/

#define GATUnknown(f,s)		(LogicalSector(f->floppy,s,f->dirtrack,GATSECTOR).data + 0xcb)
#define GATMPWHash(f,s)		(LogicalSector(f->floppy,s,f->dirtrack,GATSECTOR).data + 0xce)
#define GATDiskName(f,s)	(LogicalSector(f->floppy,s,f->dirtrack,GATSECTOR).data + 0xd0)
#define GATAuto(f,s)		(LogicalSector(f->floppy,s,f->dirtrack,GATSECTOR).data + 0xe0)
#define GATAutoRun(f,s)		(*GATAuto(f,s)==0x0d?0:1)


int	fs_trs80_init(struct floppy *,struct fs_trs80 *);
int	fs_trs80_add(struct fs_trs80 *, char *, int);
int	fs_trs80_delete(struct fs_trs80 *, char *);
int	fs_trs80_report(struct fs_trs80 *, int);
int	fs_trs80_extract(struct fs_trs80 *, char *, int);
int	fs_trs80_cleanup(struct fs_trs80 *);
char	*fs_trs80_description(void);

